#ifndef _GC_H_
#define _GC_H_

#include <CoreGc.h>
#include <LutStep.h>
#include <MathLib.h>
#include <PixelLine.h>


//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                                  TGC
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
class TGC : public TCoreGC
{
public:

  //---------------------------------------------------------------------
  //                            Constructor
  //---------------------------------------------------------------------
  TGC(){;}


  //---------------------------------------------------------------------
  //                          Initialization
  //---------------------------------------------------------------------
  void Init       ();
  void InitGrafix ();
  void InitPatn   ();
  void InitVuport ();
  void InitLut    ();
  void AmendCanvas();


  //---------------------------------------------------------------------
  //                      Image Version Control
  //---------------------------------------------------------------------
  void  SetVersions( INT32 n );
  void  Versions   ( INT32 n ){ SetVersions(n); }
  INT32 GetVersions();


  //---------------------------------------------------------------------
  //                      LUT and Video Methods
  //---------------------------------------------------------------------
  void   SetLut           ( UINT32 colorset );
  void   ResetLut         ( UINT32 colorset );
  UINT32 SetColorNamed    ( char* name );
  UINT32 SetEdgeColorNamed( char* leftname, char* rightname );
  INT32  GetEdgeColor     ( INT32 clr, INT32 sample );
  INT32  GetEdgeColorMany ();
  UINT32 RepPixel         ( INT32  clr   );
  INT32  UnRepPixel       ( UINT32 value );
  UINT16 GetColorMany     ();
  UINT16 GetColorLast     ();

  void   SetCode            ( UINT32 code    );
  void   SelectColor        ( UINT32 clr     );
  void   SetLevelRGB        ( INT32 r, INT32 g, INT32 b );
  void   SetLevelGray       ( INT32 level   );
  void   SetLevelGrayPercent( INT32  percent );
  void   SelectForeground   ();

  void   SetLevelMax       ( UINT32 max  );
  UINT32 GetLevelMax       ();
  void   SetLevelMin       ( UINT32 min  );
  UINT32 GetLevelMin       ();
  INT32  MapLevel          ( INT32 level );
  void   PokeCode          ( UINT32 code );
  void   PokeColor         ( UINT32 clr  );
  void   PokeLevelRGB      ( UINT32 r, UINT32 g, UINT32 b );
  void   PokeLevelBlankRGB ( UINT32 r, UINT32 g, UINT32 b );
  void   PokeBlackColors   ();

  UINT8  GetDigitalVideoType();
  UINT8  GetAnalogVideoType ();
  bool   DigitalVideoColor  ();
  bool   AnalogVideoColor   ();
  double GetPixelRate       ();
  UINT32 PeekRefLut         ( UINT32 index );

  //---------------------------------------------------------------------
  //                            Transparency
  //---------------------------------------------------------------------
  void Opaque     ( TColor foregnd, TColor backgnd );
  void Transparent( TColor foregnd );
 
  //---------------------------------------------------------------------
  //                            Screen Fills
  //---------------------------------------------------------------------
  void WipeShoot       ();
  void WipeActive      ();
  void ZeroActive      ();
  void GroundFillVuport()                 {WipeActive();}
  void BlackFillVuport ()                 {ZeroActive();}
  void FillActive      ( TColor clr );
  void ColorFillVuport ( TColor clr )  {FillActive( clr );}


  //---------------------------------------------------------------------
  //                         Drawing Primitives
  //---------------------------------------------------------------------
  void DrawPoint   ( INT32 x,  INT32 y );
  void DrawLine    ( INT32 x1, INT32 y1, INT32 x2,  INT32 y2 );
  void DrawRect    ( INT32 w,  INT32 h,  INT32 x,   INT32 y  );
  void DrawOval    ( INT32 w,  INT32 h,  INT32 x,   INT32 y  );
  void DottedOval  ( INT32 w,  INT32 h,  INT32 x,   INT32 y, INT32 step );
  void DrawPolyLine( INT32 nlines, POINT linelist[], POINT ptlist[] );
  void DrawTriangle( INT32 x1, INT32 y1, INT32 x2, INT32 y2,
                     INT32 x3, INT32 y3 );



  //---------------------------------------------------------------------
  //                           Drawing Composites
  //---------------------------------------------------------------------
  void DrawGrid      ( INT32 xboxes, INT32 yboxes, INT32 version );
  void DrawGrill     ( INT32 w, INT32 h, INT32 x, INT32 y, INT32 onwidth,
                       INT32 offwidth, INT32 axis );
  void DrawFormat    ( INT32 xp, INT32 yp );
  void DrawTextBox   ( INT32 xp, INT32 yp );
  void DrawTBox      ( INT32 xp, INT32 yp )             {DrawTextBox(xp, yp);}
  void DrawStep      ( INT32 xp, INT32 yp );
  void DrawFormat    ( INT32 xp, INT32 yp, TColor clr );
  void DrawTextBox   ( INT32 xp, INT32 yp, TColor clr );
  void DrawStep      ( INT32 xp, INT32 yp, TColor clr );
  void DrawCenterMark();
  void DrawLimits    ();
  void DrawCross     ();



  //---------------------------------------------------------------------
  //                          Filling Primitives
  //---------------------------------------------------------------------
  void FillRect       ( INT32 w, INT32 h, INT32 x, INT32 y );
  void FillTriangle   ( INT32 x1, INT32 y1, INT32 x2, INT32 y2,
                        INT32 x3, INT32 y3, bool solid = true );
  void FillOval       ( INT32 w, INT32 h, INT32 x, INT32 y );
  void FillPolygon    ( INT32 nvert, POINT *point, bool solid = true );

  void FrameRect      ( INT32 w, INT32 h, INT32 x, INT32 y, INT32 dx, INT32 dy );
  void FrameOval      ( INT32 w, INT32 h, INT32 xleft, INT32 ytop,
                        INT32 dx, INT32 dy, bool solid = true );

  void CurtainFillRect ( INT32 w, INT32 h, INT32 xs, INT32 ys, INT32 mode );
  void BlockCurtainFill( INT32 w, INT32 h, INT32 xs, INT32 ys, INT32 yoffset, INT32 cnt );
  void EncloseFillRect ( INT32 n, POINT ptlist[] );




  //---------------------------------------------------------------------
  //                     Pattern Filling Primitives
  //---------------------------------------------------------------------
  INT32 GetPatnMax      ();
  bool  SelectPatn      ( INT32 index );
  INT32 SelectPatn      ( const char* name );
  INT32 InstallPatn     ( INT16 *pattern );
  bool  RemovePatn      ();

  void  PatnFillRect    ( INT32 w, INT32 h, INT32 x, INT32 y );
  void  PatnFillTriangle( INT32 x1, INT32 y1, INT32 x2, INT32 y2,
                          INT32 x3,  INT32 y3 )
                                {FillTriangle( x1, y1, x2, y2, x3, y3, false );}
  void  PatnFillOval    ( INT32 w, INT32 h, INT32 x, INT32 y );
  void  PatnFillPolygon ( INT32 nvert, POINT* point)
                                {FillPolygon( nvert, point, false );}
  void  PatnFrameRect   ( INT32 w, INT32 h, INT32 x, INT32 y,
                          INT32 dx, INT32 dy );
  void  PatnFrameOval   ( INT32 w, INT32 h, INT32 xleft, INT32 ytop,
                          INT32 dx, INT32 dy )
                                 {FrameOval( w, h, xleft, ytop, dx, dy, false );}




  //---------------------------------------------------------------------
  //                  Text Rendering Primitives
  //---------------------------------------------------------------------

  //*** Character Rendering (single and block)
  INT32 DrawChar    ( INT32 x,  INT32 y,  INT32 code   );
  void  CharFillRect( INT32 width,   INT32 height,
                      INT32 left,    INT32 top,
                      INT32 spacex,  INT32 spacey,
                      INT32 marginx, INT32 marginy,
                      INT32 code, bool fast = false );

  //*** String Rendering
  INT32 DrawString  ( INT32 x,  INT32 y,  const char *string );
  void  DrawTString ( INT32 x,  INT32 y,  const char *string );
  INT32 DrawText    ( INT32 x,  INT32 y,  const char *text   );

  //*** String Bit Width
  INT32 GetWidth    ( const char* text );

  //*** DrawText() Parameters
  void  AddTextSpace( INT32 n ) {addtext = n;}
  void  AddWordSpace( INT32 n ) {addSpace = n;}
  void  AddMoreSpace( INT32 n ) {modSpace = n;}
  INT32 GetTextSpace()          {return addtext;}
  INT32 GetWordSpace()          {return addSpace;}
  INT32 GetMoreSpace()          {return modSpace;}



  //---------------------------------------------------------------------
  //             Font Management
  //---------------------------------------------------------------------

  //*** Font Installation and Selection
  void  DefaultFont      ();
  bool  SelectFont       ( UINT32 index );
  INT32 SelectFont       ( const char* fontname );
  INT32 SelectFontNamed  ( const char* fontname )
                                                {return SelectFont( fontname );}
  bool  GetFontName      ( char* name );
  INT32 GetFontIndex     ( const char* fontname )
                                           {return FontNameToIndex( fontname );}

  //*** Parameters for the Currently Selected Font
  INT32 GetAscent        ();
  INT32 GetDescent       ();
  INT32 GetLeading       ();
  INT32 CharWideMax      ();
  INT32 CharHigh         ();
  INT32 GetCharWidthImage( INT32 code );
  INT32 CharWideImage    ( INT32 code )      {return GetCharWidthImage( code );}
  INT32 GetFirstCh       ();
  INT32 GetLastCh        ();
  INT32 GetFontMax       ();



  //---------------------------------------------------------------------
  //          2-D Methods Utilizing the 'ctm' Global Matrix
  //---------------------------------------------------------------------
  INT32  ScaleX     ( INT32  h  );
  INT32  ScaleY     ( INT32  w  );
  INT32  SquareDX   ( INT32  h  )                          {return ScaleX( h );}
  INT32  SquareDY   ( INT32  w  )                          {return ScaleY( w );}
  INT32  TransformX ( double x  );
  INT32  TransformY ( double y  );
  double FTransformX( double x  );
  double FTransformY( double y  );
  INT32  TransformDX( double dx );
  INT32  TransformDY( double dy );
  void   TranslateX ( double x  );
  void   TranslateY ( double y  );


  //---------------------------------------------------------------------
  //                          3-D Methods
  //---------------------------------------------------------------------
  void InitMatrix ( FIX* matrix );
  void ResetMatrix();
  void CopyMatrix ( FIX* inmatrix, FIX* outmatrix );
  void CopyVertex ( INT32 nverts, FIX* invertex, FIX* outvertex );
  void CreateIdentity( double Mat[3][3] );
  void MatrixMult    ( double MatA[3][3], double MatB[3][3],
                       double Matfinal[3][3] );
  void Translate  ( FIX* matrix, FIX* displacement );
  void Transform  ( FIX* matrix, INT32 n, FIX* verts );
  void Scale      ( FIX* matrix, FIX* factor );
  void Rotate     ( FIX* matrix, FIX* angle );
  void Perspec    ( INT32 n, FIX* verts, POINT* ptlist,
                    INT32 x, INT32 y, INT32 z );


  //---------------------------------------------------------------------
  //                    Windowing and Clipping
  //---------------------------------------------------------------------
  INT32 GetVuportMax ();
  INT32 OpenVuport   ();
  INT32 CloseVuport  ( INT32 index );
  INT32 RemoveVuport ( INT32 index );
  INT32 SelectVuport ( INT32 index );
  INT32 CopyVuport   (INT32 index1, INT32 index2);
  INT32 CurrentVuport();
  void  SetOrigin    ( INT32 x, INT32 y );
  INT32 SizeVuport   ( INT32 w, INT32 h );
  void  MoveVuport   ( INT32 x, INT32 y );
  INT32 SetClipRect  ( INT32 w, INT32 h, INT32 x, INT32 y );
  INT32 PhSizeVuport ( double wSize, double hSize );



  //---------------------------------------------------------------------
  //                     Bitmap\Icon Primitives
  //---------------------------------------------------------------------

  void   PixelExpand( UINT32* srcmap, UINT32 srcpitch32, INT32 srcpsize,
                      INT32 w, INT32 h, INT32 xd, INT32 yd );
  TColor GetPixel ( INT32 xd, INT32 yd );
  void   GetRect  ( INT32 w, INT32 h, INT32 xs, INT32 ys, UINT32* array, UINT32 pitch32 );

  void   PutPixel ( TColor clr, INT32 xd, INT32 yd );
  void   PutRect  ( UINT32* array, UINT32 pitch32, INT32 w, INT32 h, INT32 xd, INT32 yd );

  void   ZoomRect ( INT32 ws, INT32 hs, INT32 xs, INT32 ys, INT32 wd, INT32 hd,
                    INT32 xd, INT32 yd, INT16* array, bool foregrnd = false );

  void   MoveRect ( INT32 w, INT32 h, INT32 xs, INT32 ys, INT32 xd, INT32 yd );


  //---------------------------------------------------------------------
  //                            Animation
  //---------------------------------------------------------------------
  void   WaitScan       ( INT32  y      );
  UINT32 WaitFrame      ( UINT32 frame  );
  UINT32 Delay          ( INT32  frames );
  void   SetDisplayMask ( UINT32 mask   );
  void   SetDisplayStart( INT32  y      );
  void   SetDisplayDelta( bool   normal );
  #define NORMAL true
  #define REPEAT false
  void   SetFrameSync   ( UINT32 frame  );
  void   SetFrameNext   ( UINT32 frame  );
  UINT32 GetDisplayMask ();
  UINT32 GetFrame       ();
  UINT32 GetFrameNext   ();
  UINT32 GetFrameSync   ();

  //*** Is it ok to proceed?
  bool Proceed() const;

  //---------------------------------------------------------------------
  //                        Utility Methods
  //---------------------------------------------------------------------
  void   Boxes( INT32* xbox, INT32* ybox, INT32 density, INT32 even );

  //---------------------------------------------------------------------
  //                         User Parameters
  //---------------------------------------------------------------------

  INT32 GetPenW      () const;
  INT32 GetPenH      () const;
  INT32 GetDwell     () const;
  INT32 GetOffsetX   () const;
  INT32 GetOffsetY   () const;
  INT32 GetSpaceX    () const;
  INT32 GetSpaceY    () const;
  INT32 GetDeltaX    () const;
  INT32 GetDeltaY    () const;
  INT32 GetCycles    () const;
  INT32 GetLevelred  () const;
  INT32 GetLevelgreen() const;
  INT32 GetLevelblue () const;

  void SetPenW      ( INT32 value );
  void SetPenH      ( INT32 value );
  void SetDwell     ( INT32 value );
  void SetOffsetX   ( INT32 value );
  void SetOffsetY   ( INT32 value );
  void SetSpaceX    ( INT32 value );
  void SetSpaceY    ( INT32 value );
  void SetDeltaX    ( INT32 value );
  void SetDeltaY    ( INT32 value );
  void SetCycles    ( INT32 value );
  void SetLevelred  ( INT32 value );
  void SetLevelgreen( INT32 value );
  void SetLevelblue ( INT32 value );

  //---------------------------------------------------------------------
  //                Special Line Rendering Methods
  //---------------------------------------------------------------------

  bool InitLine  ( TPixelLine& line );
  bool RenderLine( TPixelLine& line, INT32 startY, INT32 endY );

  void L1t5pb   ( TPixelLine& line );
  void SquareSignal( TPixelLine& line, double freq );
  void L2t5pb   ( TPixelLine& line );
  void FiveStep ( TPixelLine& line );
  void Bars100  ( TPixelLine& line );
  void Bars75   ( TPixelLine& line );
  void BlokDelay( TPixelLine& line );
  void Bowtie1ns( TPixelLine& line );
  void Bowtie2ns( TPixelLine& line );
  void BowtieMarkers( TPixelLine& line );
  void KaderH( TPixelLine& line );
  void KaderV( TPixelLine& line );
  void Kader ( TPixelLine& line );
  void LSweep( TPixelLine& line, double startFreq, double endFreq, double factor );
  void Markers ( TPixelLine& line, double factor );
  void Mburst  ( TPixelLine& line, double peakLevel );
  void Notch   ( TPixelLine& line, double freq );
  void PulseAndBar1( TPixelLine& line );
  void PulseAndBar2( TPixelLine& line );
  void Ramp( TPixelLine& line );

};


#endif






